using System;
using System.Collections;
using UnityEngine;
using UnityEngine.Networking;
using XenseAR;

public class AnchoringDetails : MonoBehaviour
{
    public string sizeStr = "NaN";
    public Color dataSizeBackgroundColor = Color.green;
    string anchorUrl = "";
    void Start()
    {
        LoadHeaderInfo();
    }

    public void Refresh()
    {
        LoadHeaderInfo();
    }

    private void LoadHeaderInfo()
    {
        if(gameObject.GetComponent<BaseAnchor>() != null)
        {
            anchorUrl = gameObject.GetComponent<BaseAnchor>().GetAnchoringTarget().media.url;
        }

        if(string.IsNullOrEmpty(anchorUrl))
        {
            Debug.Log("Game object is not valid");
            return;
        }

        StartCoroutine(GetContentLength(anchorUrl, (size) =>
        {
            if (size >= 0)
            {
                try
                {
                    if (size / 1024 > 1024) {
                        sizeStr = "" + size / 1024 /1024 + "MB";
                        if(size / 1024 /1024 > 10)
                            dataSizeBackgroundColor = Color.red;
                        else dataSizeBackgroundColor = Color.yellow;
                    } else
                    {
                        sizeStr = "" + size / 1024 + "KB";
                        dataSizeBackgroundColor = Color.green;
                    }
                }
                catch (Exception e)
                {
                    Debug.Log("Error: Cannot parse");
                }
                
                Debug.Log("File size (bytes): " + sizeStr);
            }
            else
                Debug.Log("Could not get size or no Content-Length header");
        }));
    }

    IEnumerator GetContentLength(string url, System.Action<long> callback)
    {
        UnityWebRequest uwr = UnityWebRequest.Head(url);
        yield return uwr.SendWebRequest();

        if (uwr.result != UnityWebRequest.Result.Success)
        {
            //Debug.LogError("Error getting header: " + uwr.error);
            StartCoroutine(RetryGetContentLength(url, callback));
        }
        else
        {
            string lengthHeader = uwr.GetResponseHeader("Content-Length");
            if (long.TryParse(lengthHeader, out long size))
                callback(size);
            else
                callback(-1);
        }
    }

    IEnumerator RetryGetContentLength(string url, System.Action<long> callback)
    {
        if (gameObject.GetComponent<BaseAnchor>() is AssetbundleAnchorEditor)
        {
            #if UNITY_EDITOR_WIN
                string bundleURL_post = url + Defines.androidPostfix;
            #elif UNITY_EDITOR_OSX
                string bundleURL_post = url + Defines.iosPostfix;
            #else
            #if UNITY_ANDROID
                string bundleURL_post = url + Defines.androidPostfix;
            #endif
            #if UNITY_IOS
                string bundleURL_post = url + Defines.iosPostfix;
            #endif
            #if UNITY_WEBGL
                string bundleURL_post = url + Defines.webglPostfix;
            #endif
            #if UNITY_WSA
                string bundleURL_post = url + Defines.wsaPlayerPostfix;
            #endif
            #if UNITY_STANDALONE_WIN
                string bundleURL_post = url + Defines.standaloneWindows64Postfix;
            #endif
            #if UNITY_STANDALONE_OSX
                string bundleURL_post = url + Defines.standaloneOSXPostfix;
            #endif
            #endif
            anchorUrl = bundleURL_post;

            UnityWebRequest uwr = UnityWebRequest.Head(anchorUrl);
            yield return uwr.SendWebRequest();

            if (uwr.result != UnityWebRequest.Result.Success)
            {
                Debug.LogError("Error getting header: " + uwr.error);
                callback(-1);
            }
            else
            {
                string lengthHeader = uwr.GetResponseHeader("Content-Length");
                if (long.TryParse(lengthHeader, out long size))
                    callback(size);
                else
                    callback(-1);
            }
        }
    }
}
